/*
 * Decompiled with CFR 0.152.
 */
package mekanism.common.lib.radiation;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import com.google.common.collect.Tables;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import mekanism.api.Chunk3D;
import mekanism.api.annotations.NothingNullByDefault;
import mekanism.api.chemical.gas.GasStack;
import mekanism.api.chemical.gas.IGasHandler;
import mekanism.api.chemical.gas.IGasTank;
import mekanism.api.chemical.gas.attribute.GasAttributes;
import mekanism.api.math.MathUtils;
import mekanism.api.radiation.IRadiationManager;
import mekanism.api.radiation.IRadiationSource;
import mekanism.api.radiation.capability.IRadiationEntity;
import mekanism.api.radiation.capability.IRadiationShielding;
import mekanism.api.text.EnumColor;
import mekanism.common.capabilities.Capabilities;
import mekanism.common.config.MekanismConfig;
import mekanism.common.lib.MekanismSavedData;
import mekanism.common.lib.collection.HashList;
import mekanism.common.lib.radiation.Meltdown;
import mekanism.common.lib.radiation.RadiationSource;
import mekanism.common.network.PacketUtils;
import mekanism.common.network.to_client.radiation.PacketEnvironmentalRadiationData;
import mekanism.common.network.to_client.radiation.PacketPlayerRadiationData;
import mekanism.common.registries.MekanismDamageTypes;
import mekanism.common.registries.MekanismParticleTypes;
import mekanism.common.registries.MekanismSounds;
import mekanism.common.util.EnumUtils;
import mekanism.common.util.MekanismUtils;
import net.minecraft.core.BlockPos;
import net.minecraft.core.GlobalPos;
import net.minecraft.core.RegistryAccess;
import net.minecraft.core.Vec3i;
import net.minecraft.core.particles.ParticleOptions;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.Tag;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.sounds.SoundEvent;
import net.minecraft.util.Mth;
import net.minecraft.util.RandomSource;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.damagesource.DamageType;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EquipmentSlot;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.Level;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.neoforge.event.entity.living.LivingEvent;
import net.neoforged.neoforge.server.ServerLifecycleHooks;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

@NothingNullByDefault
public class RadiationManager
implements IRadiationManager {
    private static final String DATA_HANDLER_NAME = "radiation_manager";
    private static final RandomSource RAND = RandomSource.create();
    public static final double BASELINE = 1.0E-7;
    public static final double MIN_MAGNITUDE = 1.0E-5;
    private boolean loaded;
    private final Table<Chunk3D, GlobalPos, RadiationSource> radiationTable = HashBasedTable.create();
    private final Table<Chunk3D, GlobalPos, IRadiationSource> radiationView = Tables.unmodifiableTable(this.radiationTable);
    private final Map<ResourceLocation, List<Meltdown>> meltdowns = new Object2ObjectOpenHashMap();
    private final Map<UUID, PreviousRadiationData> playerEnvironmentalExposureMap = new Object2ObjectOpenHashMap();
    private final Map<UUID, PreviousRadiationData> playerExposureMap = new Object2ObjectOpenHashMap();
    private RadiationScale clientRadiationScale = RadiationScale.NONE;
    private double clientEnvironmentalRadiation = 1.0E-7;
    private double clientMaxMagnitude = 1.0E-7;
    @Nullable
    private RadiationDataHandler dataHandler;

    public static RadiationManager get() {
        return (RadiationManager)INSTANCE;
    }

    @Override
    public boolean isRadiationEnabled() {
        return MekanismConfig.general.radiationEnabled.getOrDefault();
    }

    private void markDirty() {
        if (this.dataHandler != null) {
            this.dataHandler.setDirty();
        }
    }

    @Override
    public DamageSource getRadiationDamageSource(RegistryAccess registryAccess) {
        return MekanismDamageTypes.RADIATION.source(registryAccess);
    }

    @Override
    public ResourceKey<DamageType> getRadiationDamageTypeKey() {
        return MekanismDamageTypes.RADIATION.key();
    }

    @Override
    public double getRadiationLevel(Entity entity) {
        if (this.radiationTable.isEmpty()) {
            return 1.0E-7;
        }
        return this.getRadiationLevel(GlobalPos.of((ResourceKey)entity.level().dimension(), (BlockPos)entity.blockPosition()));
    }

    public int getDecayTime(double magnitude, boolean source) {
        double decayRate = source ? MekanismConfig.general.radiationSourceDecayRate.get() : MekanismConfig.general.radiationTargetDecayRate.get();
        int seconds = 0;
        double localMagnitude = magnitude;
        while (localMagnitude > 1.0E-5) {
            localMagnitude *= decayRate;
            ++seconds;
        }
        return seconds;
    }

    @Override
    public Table<Chunk3D, GlobalPos, IRadiationSource> getRadiationSources() {
        return this.radiationView;
    }

    @Override
    public void removeRadiationSources(Chunk3D chunk) {
        Map chunkSources = this.radiationTable.row((Object)chunk);
        if (!chunkSources.isEmpty()) {
            chunkSources.clear();
            this.markDirty();
            this.updateClientRadiationForAll(chunk.dimension);
        }
    }

    @Override
    public void removeRadiationSource(GlobalPos pos) {
        Chunk3D chunk = new Chunk3D(pos);
        if (this.radiationTable.contains((Object)chunk, (Object)pos)) {
            this.radiationTable.remove((Object)chunk, (Object)pos);
            this.markDirty();
            this.updateClientRadiationForAll((ResourceKey<Level>)pos.dimension());
        }
    }

    @Override
    public double getRadiationLevel(GlobalPos pos) {
        if (this.radiationTable.isEmpty()) {
            return 1.0E-7;
        }
        return this.getRadiationLevelAndMaxMagnitude(pos).level();
    }

    public LevelAndMaxMagnitude getRadiationLevelAndMaxMagnitude(Entity entity) {
        if (this.radiationTable.isEmpty()) {
            return LevelAndMaxMagnitude.BASELINE;
        }
        return this.getRadiationLevelAndMaxMagnitude(GlobalPos.of((ResourceKey)entity.level().dimension(), (BlockPos)entity.blockPosition()));
    }

    public LevelAndMaxMagnitude getRadiationLevelAndMaxMagnitude(GlobalPos pos) {
        if (this.radiationTable.isEmpty()) {
            return LevelAndMaxMagnitude.BASELINE;
        }
        double level = 1.0E-7;
        double maxMagnitude = 1.0E-7;
        Chunk3D center = new Chunk3D(pos);
        int radius = MekanismConfig.general.radiationChunkCheckRadius.get();
        double maxRange = Mth.square((int)(radius * 16));
        int minX = center.x - radius;
        int maxX = center.x + radius;
        int minZ = center.z - radius;
        int maxZ = center.z + radius;
        for (int i = minX; i <= maxX; ++i) {
            for (int j = minZ; j <= maxZ; ++j) {
                Chunk3D chunk = new Chunk3D(center.dimension, i, j);
                for (Map.Entry entry : this.radiationTable.row((Object)chunk).entrySet()) {
                    if (!(((GlobalPos)entry.getKey()).pos().distSqr((Vec3i)pos.pos()) <= maxRange)) continue;
                    RadiationSource source = (RadiationSource)entry.getValue();
                    level += this.computeExposure(pos, source);
                    maxMagnitude = Math.max(maxMagnitude, source.getMagnitude());
                }
            }
        }
        return new LevelAndMaxMagnitude(level, maxMagnitude);
    }

    @Override
    public void radiate(GlobalPos pos, double magnitude) {
        if (!this.isRadiationEnabled()) {
            return;
        }
        Map radiationSourceMap = this.radiationTable.row((Object)new Chunk3D(pos));
        RadiationSource src = (RadiationSource)radiationSourceMap.get(pos);
        if (src == null) {
            radiationSourceMap.put(pos, new RadiationSource(pos, magnitude));
        } else {
            src.radiate(magnitude);
        }
        this.markDirty();
        this.updateClientRadiationForAll((ResourceKey<Level>)pos.dimension());
    }

    @Override
    public void radiate(LivingEntity entity, double magnitude) {
        IRadiationEntity radiationEntity;
        Player player;
        if (!this.isRadiationEnabled()) {
            return;
        }
        if ((!(entity instanceof Player) || MekanismUtils.isPlayingMode(player = (Player)entity)) && (radiationEntity = (IRadiationEntity)entity.getCapability(Capabilities.RADIATION_ENTITY)) != null) {
            radiationEntity.radiate(magnitude * (1.0 - Math.min(1.0, this.getRadiationResistance(entity))));
        }
    }

    @Override
    public void dumpRadiation(GlobalPos pos, IGasHandler gasHandler, boolean clearRadioactive) {
        int gasTanks = gasHandler.getTanks();
        for (int tank = 0; tank < gasTanks; ++tank) {
            if (!this.dumpRadiation(pos, (GasStack)gasHandler.getChemicalInTank(tank)) || !clearRadioactive) continue;
            gasHandler.setChemicalInTank(tank, GasStack.EMPTY);
        }
    }

    @Override
    public void dumpRadiation(GlobalPos pos, List<IGasTank> gasTanks, boolean clearRadioactive) {
        for (IGasTank gasTank : gasTanks) {
            if (!this.dumpRadiation(pos, (GasStack)gasTank.getStack()) || !clearRadioactive) continue;
            gasTank.setEmpty();
        }
    }

    @Override
    public boolean dumpRadiation(GlobalPos pos, GasStack stack) {
        double radioactivity;
        if (this.isRadiationEnabled() && !stack.isEmpty() && (radioactivity = stack.mapAttributeToDouble(GasAttributes.Radiation.class, (stored, attribute) -> (double)stored.getAmount() * attribute.getRadioactivity())) > 0.0) {
            this.radiate(pos, radioactivity);
            return true;
        }
        return false;
    }

    public void createMeltdown(Level world, BlockPos minPos, BlockPos maxPos, double magnitude, double chance, float radius, UUID multiblockID) {
        this.meltdowns.computeIfAbsent(world.dimension().location(), id -> new ArrayList()).add(new Meltdown(minPos, maxPos, magnitude, chance, radius, multiblockID));
        this.markDirty();
    }

    public void clearSources() {
        if (!this.radiationTable.isEmpty()) {
            this.radiationTable.clear();
            this.markDirty();
            this.updateClientRadiationForAll();
        }
    }

    private double computeExposure(GlobalPos pos, RadiationSource source) {
        return source.getMagnitude() / Math.max(1.0, pos.pos().distSqr((Vec3i)source.getPos().pos()));
    }

    private double getRadiationResistance(LivingEntity entity) {
        double resistance = 0.0;
        for (EquipmentSlot type : EnumUtils.ARMOR_SLOTS) {
            IRadiationShielding shielding;
            ItemStack stack = entity.getItemBySlot(type);
            if (stack.isEmpty() || (shielding = (IRadiationShielding)stack.getCapability(Capabilities.RADIATION_SHIELDING)) == null) continue;
            resistance += shielding.getRadiationShielding();
        }
        return resistance;
    }

    private void updateClientRadiationForAll(ResourceKey<Level> dimension) {
        MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
        if (server != null) {
            for (ServerPlayer player : server.getPlayerList().getPlayers()) {
                if (player.level().dimension() != dimension) continue;
                this.updateClientRadiation(player);
            }
        }
    }

    private void updateClientRadiationForAll() {
        MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
        if (server != null) {
            for (ServerPlayer player : server.getPlayerList().getPlayers()) {
                this.updateClientRadiation(player);
            }
        }
    }

    public void updateClientRadiation(ServerPlayer player) {
        LevelAndMaxMagnitude levelAndMaxMagnitude = this.getRadiationLevelAndMaxMagnitude((Entity)player);
        PreviousRadiationData previousRadiationData = this.playerEnvironmentalExposureMap.get(player.getUUID());
        PreviousRadiationData relevantData = PreviousRadiationData.compareTo(previousRadiationData, levelAndMaxMagnitude.level());
        if (relevantData != null) {
            this.playerEnvironmentalExposureMap.put(player.getUUID(), relevantData);
            PacketUtils.sendTo(new PacketEnvironmentalRadiationData(levelAndMaxMagnitude), player);
        }
    }

    public void setClientEnvironmentalRadiation(double radiation, double maxMagnitude) {
        this.clientEnvironmentalRadiation = radiation;
        this.clientMaxMagnitude = maxMagnitude;
        this.clientRadiationScale = RadiationScale.get(this.clientEnvironmentalRadiation);
    }

    public double getClientEnvironmentalRadiation() {
        return this.isRadiationEnabled() ? this.clientEnvironmentalRadiation : 1.0E-7;
    }

    public double getClientMaxMagnitude() {
        return this.isRadiationEnabled() ? this.clientMaxMagnitude : 1.0E-7;
    }

    public RadiationScale getClientScale() {
        return this.isRadiationEnabled() ? this.clientRadiationScale : RadiationScale.NONE;
    }

    public void tickClient(Player player) {
        if (!this.isRadiationEnabled()) {
            return;
        }
        RandomSource randomSource = player.level().getRandom();
        if (this.clientRadiationScale != RadiationScale.NONE && MekanismConfig.client.radiationParticleCount.get() != 0 && randomSource.nextInt(2) == 0) {
            int count = randomSource.nextInt(this.clientRadiationScale.ordinal() * MekanismConfig.client.radiationParticleCount.get());
            int radius = MekanismConfig.client.radiationParticleRadius.get();
            for (int i = 0; i < count; ++i) {
                double x = player.getX() + randomSource.nextDouble() * (double)radius * 2.0 - (double)radius;
                double y = player.getY() + randomSource.nextDouble() * (double)radius * 2.0 - (double)radius;
                double z = player.getZ() + randomSource.nextDouble() * (double)radius * 2.0 - (double)radius;
                player.level().addParticle((ParticleOptions)MekanismParticleTypes.RADIATION.get(), x, y, z, 0.0, 0.0, 0.0);
            }
        }
    }

    public void tickServer(ServerPlayer player) {
        this.updateEntityRadiation((LivingEntity)player);
    }

    private void updateEntityRadiation(LivingEntity entity) {
        if (!this.isRadiationEnabled()) {
            return;
        }
        IRadiationEntity radiationCap = (IRadiationEntity)entity.getCapability(Capabilities.RADIATION_ENTITY);
        if (entity.level().getRandom().nextInt(20) == 0) {
            Player player;
            double magnitude = this.getRadiationLevel((Entity)entity);
            if (magnitude > 1.0E-7 && (!(entity instanceof Player) || MekanismUtils.isPlayingMode(player = (Player)entity))) {
                this.radiate(entity, magnitude / 3600.0);
            }
            if (radiationCap != null) {
                radiationCap.decay();
            }
        }
        if (radiationCap != null) {
            radiationCap.update();
            if (entity instanceof ServerPlayer) {
                ServerPlayer player = (ServerPlayer)entity;
                double radiation = radiationCap.getRadiation();
                PreviousRadiationData previousRadiationData = this.playerExposureMap.get(player.getUUID());
                PreviousRadiationData relevantData = PreviousRadiationData.compareTo(previousRadiationData, radiation);
                if (relevantData != null) {
                    this.playerExposureMap.put(player.getUUID(), relevantData);
                    PacketUtils.sendTo(new PacketPlayerRadiationData(radiation), player);
                }
            }
        }
    }

    public void tickServerWorld(ServerLevel world) {
        List dimensionMeltdowns;
        if (!this.isRadiationEnabled()) {
            return;
        }
        if (!this.loaded) {
            this.createOrLoad();
        }
        if (!(dimensionMeltdowns = this.meltdowns.getOrDefault(world.dimension().location(), Collections.emptyList())).isEmpty()) {
            Iterator iterator = dimensionMeltdowns.iterator();
            while (iterator.hasNext()) {
                Meltdown meltdown = (Meltdown)iterator.next();
                if (!meltdown.update(world)) continue;
                iterator.remove();
            }
            this.markDirty();
        }
    }

    public void tickServer() {
        Collection sources;
        if (!this.isRadiationEnabled() || this.radiationTable.isEmpty()) {
            return;
        }
        if (RAND.nextInt(20) == 0 && !(sources = this.radiationTable.values()).isEmpty()) {
            sources.removeIf(RadiationSource::decay);
            this.markDirty();
            this.updateClientRadiationForAll();
        }
    }

    public void createOrLoad() {
        if (this.dataHandler == null) {
            this.dataHandler = MekanismSavedData.createSavedData(RadiationDataHandler::new, DATA_HANDLER_NAME);
            this.dataHandler.setManagerAndSync(this);
            this.dataHandler.clearCached();
        }
        this.loaded = true;
    }

    public void reset() {
        this.radiationTable.clear();
        this.playerEnvironmentalExposureMap.clear();
        this.playerExposureMap.clear();
        this.meltdowns.clear();
        this.dataHandler = null;
        this.loaded = false;
    }

    public void resetClient() {
        this.setClientEnvironmentalRadiation(1.0E-7, 1.0E-7);
    }

    public void resetPlayer(UUID uuid) {
        this.playerEnvironmentalExposureMap.remove(uuid);
        this.playerExposureMap.remove(uuid);
    }

    @SubscribeEvent
    public void onLivingTick(LivingEvent.LivingTickEvent event) {
        Level world = event.getEntity().level();
        if (!world.isClientSide() && !(event.getEntity() instanceof Player)) {
            this.updateEntityRadiation(event.getEntity());
        }
    }

    public static enum RadiationScale {
        NONE,
        LOW,
        MEDIUM,
        ELEVATED,
        HIGH,
        EXTREME;

        private static final double LOG_BASELINE;
        private static final double LOG_MAX;
        private static final double SCALE;

        public static RadiationScale get(double magnitude) {
            if (magnitude < 1.0E-5) {
                return NONE;
            }
            if (magnitude < 0.001) {
                return LOW;
            }
            if (magnitude < 0.1) {
                return MEDIUM;
            }
            if (magnitude < 10.0) {
                return ELEVATED;
            }
            if (magnitude < 100.0) {
                return HIGH;
            }
            return EXTREME;
        }

        public static EnumColor getSeverityColor(double magnitude) {
            if (magnitude <= 1.0E-7) {
                return EnumColor.BRIGHT_GREEN;
            }
            if (magnitude < 1.0E-5) {
                return EnumColor.GRAY;
            }
            if (magnitude < 0.001) {
                return EnumColor.YELLOW;
            }
            if (magnitude < 0.1) {
                return EnumColor.ORANGE;
            }
            if (magnitude < 10.0) {
                return EnumColor.RED;
            }
            return EnumColor.DARK_RED;
        }

        public static double getScaledDoseSeverity(double magnitude) {
            if (magnitude < 1.0E-5) {
                return 0.0;
            }
            return Math.min(1.0, Math.max(0.0, (-LOG_BASELINE + Math.log10(magnitude)) / SCALE));
        }

        public SoundEvent getSoundEvent() {
            return switch (this) {
                case LOW -> (SoundEvent)MekanismSounds.GEIGER_SLOW.get();
                case MEDIUM -> (SoundEvent)MekanismSounds.GEIGER_MEDIUM.get();
                case ELEVATED, HIGH -> (SoundEvent)MekanismSounds.GEIGER_ELEVATED.get();
                case EXTREME -> (SoundEvent)MekanismSounds.GEIGER_FAST.get();
                default -> null;
            };
        }

        static {
            LOG_BASELINE = Math.log10(1.0E-5);
            LOG_MAX = Math.log10(100.0);
            SCALE = LOG_MAX - LOG_BASELINE;
        }
    }

    public static class RadiationDataHandler
    extends MekanismSavedData {
        private Map<ResourceLocation, List<Meltdown>> savedMeltdowns = Collections.emptyMap();
        public Set<RadiationSource> loadedSources = Collections.emptySet();
        @Nullable
        public RadiationManager manager;

        public void setManagerAndSync(RadiationManager m) {
            this.manager = m;
            if (IRadiationManager.INSTANCE.isRadiationEnabled()) {
                for (RadiationSource radiationSource : this.loadedSources) {
                    this.manager.radiationTable.put((Object)new Chunk3D(radiationSource.getPos()), (Object)radiationSource.getPos(), (Object)radiationSource);
                }
                for (Map.Entry entry : this.savedMeltdowns.entrySet()) {
                    List<Meltdown> meltdowns = this.manager.meltdowns.get(entry.getKey());
                    if (meltdowns == null) {
                        this.manager.meltdowns.put((ResourceLocation)entry.getKey(), new ArrayList((Collection)entry.getValue()));
                        continue;
                    }
                    meltdowns.addAll((Collection)entry.getValue());
                }
            }
        }

        public void clearCached() {
            this.loadedSources = Collections.emptySet();
            this.savedMeltdowns = Collections.emptyMap();
        }

        @Override
        public void load(@NotNull CompoundTag nbtTags) {
            if (nbtTags.contains("radList", 9)) {
                ListTag list = nbtTags.getList("radList", 10);
                this.loadedSources = new HashList<RadiationSource>();
                for (Tag nbt2 : list) {
                    RadiationSource.load((CompoundTag)nbt2).ifPresent(this.loadedSources::add);
                }
            } else {
                this.loadedSources = Collections.emptySet();
            }
            if (nbtTags.contains("meltdowns", 10)) {
                CompoundTag meltdownNBT = nbtTags.getCompound("meltdowns");
                this.savedMeltdowns = new HashMap<ResourceLocation, List<Meltdown>>(meltdownNBT.size());
                for (String dim : meltdownNBT.getAllKeys()) {
                    ResourceLocation dimension = ResourceLocation.tryParse((String)dim);
                    if (dimension == null) continue;
                    ListTag meltdowns = meltdownNBT.getList(dim, 10);
                    this.savedMeltdowns.put(dimension, meltdowns.stream().map(nbt -> Meltdown.load((CompoundTag)nbt)).collect(Collectors.toList()));
                }
            } else {
                this.savedMeltdowns = Collections.emptyMap();
            }
        }

        @NotNull
        public CompoundTag save(@NotNull CompoundTag nbtTags) {
            if (this.manager != null && !this.manager.radiationTable.isEmpty()) {
                ListTag list = new ListTag();
                for (RadiationSource radiationSource : this.manager.radiationTable.values()) {
                    list.add((Object)radiationSource.write());
                }
                nbtTags.put("radList", (Tag)list);
            }
            if (this.manager != null && !this.manager.meltdowns.isEmpty()) {
                CompoundTag meltdownNBT = new CompoundTag();
                for (Map.Entry entry : this.manager.meltdowns.entrySet()) {
                    List meltdowns = (List)entry.getValue();
                    if (meltdowns.isEmpty()) continue;
                    ListTag list = new ListTag();
                    for (Meltdown meltdown : meltdowns) {
                        CompoundTag compound = new CompoundTag();
                        meltdown.write(compound);
                        list.add((Object)compound);
                    }
                    meltdownNBT.put(((ResourceLocation)entry.getKey()).toString(), (Tag)list);
                }
                if (!meltdownNBT.isEmpty()) {
                    nbtTags.put("meltdowns", (Tag)meltdownNBT);
                }
            }
            return nbtTags;
        }
    }

    public record LevelAndMaxMagnitude(double level, double maxMagnitude) {
        private static final LevelAndMaxMagnitude BASELINE = new LevelAndMaxMagnitude(1.0E-7, 1.0E-7);
    }

    private record PreviousRadiationData(double magnitude, int power, double base) {
        private static int getPower(double magnitude) {
            return MathUtils.clampToInt(Math.floor(Math.log10(magnitude)));
        }

        @Nullable
        private static PreviousRadiationData compareTo(@Nullable PreviousRadiationData previousRadiationData, double magnitude) {
            int power;
            if (previousRadiationData == null || Math.abs(magnitude - previousRadiationData.magnitude) >= previousRadiationData.base) {
                return PreviousRadiationData.getData(magnitude, PreviousRadiationData.getPower(magnitude));
            }
            if (magnitude < previousRadiationData.magnitude && (power = PreviousRadiationData.getPower(magnitude)) < previousRadiationData.power) {
                return PreviousRadiationData.getData(magnitude, power);
            }
            return null;
        }

        private static PreviousRadiationData getData(double magnitude, int power) {
            int siPower = Math.floorDiv(power, 3) * 3;
            double base = Math.pow(10.0, siPower - 2);
            return new PreviousRadiationData(magnitude, power, base);
        }
    }
}

