/*
 * Decompiled with CFR 0.152.
 */
package net.creeperhost.minetogether.session;

import com.google.gson.Gson;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import javax.annotation.WillClose;
import net.creeperhost.minetogether.session.HttpUtils;
import net.creeperhost.minetogether.session.InvalidJWebToken;
import net.creeperhost.minetogether.session.JWebToken;
import net.creeperhost.minetogether.session.SessionProvider;
import net.creeperhost.minetogether.session.SessionValidationException;
import net.creeperhost.minetogether.session.crypto.Crypto;
import net.creeperhost.minetogether.session.crypto.ECUtils;
import net.creeperhost.minetogether.session.crypto.PEMUtils;
import net.creeperhost.minetogether.session.crypto.RSAUtils;
import net.creeperhost.minetogether.session.data.AuthRequest;
import net.creeperhost.minetogether.session.data.AuthResponse;
import net.creeperhost.minetogether.session.data.mc.ProfileKeyPairResponse;
import org.jetbrains.annotations.Nullable;

public final class MineTogetherSession {
    private static final Gson GSON = new Gson();
    private static final MineTogetherSession DEFAULT = new MineTogetherSession(Paths.get("./.mtsession", new String[0]));
    private final ExecutorService SESSION_EXECUTOR = Executors.newSingleThreadExecutor(r -> {
        Thread th = new Thread(r);
        th.setDaemon(true);
        th.setName("MineTogetherSession Executor");
        return th;
    });
    private final Path sessionStorage;
    @Nullable
    private SessionProvider provider;
    private final List<Consumer<JWebToken>> onRefreshedCallbacks = new ArrayList<Consumer<JWebToken>>();
    @Nullable
    private JWebToken token;
    @Nullable
    private CompletableFuture<JWebToken> tokenFuture;

    private MineTogetherSession(Path sessionStorage) {
        this.sessionStorage = sessionStorage;
    }

    public static MineTogetherSession createCustom(Path storagePath) {
        return new MineTogetherSession(storagePath);
    }

    public static MineTogetherSession getDefault() {
        return DEFAULT;
    }

    public void setProvider(SessionProvider provider) {
        if (this.provider != null) {
            this.provider.infoLog("Session provider already set to " + this.provider.getClass() + " Tried to set to " + provider.getClass() + ". This will be ignired.", new Object[0]);
            return;
        }
        this.provider = provider;
    }

    public boolean isOffline() {
        if (this.provider == null) {
            return true;
        }
        UUID uuid = this.provider.getUUID();
        return uuid == null || uuid.version() != 4;
    }

    public boolean isValid() {
        if (this.provider == null) {
            return false;
        }
        if (this.token == null) {
            return false;
        }
        return this.token.getUuid().equals(this.provider.getUUID()) && this.token.isValid(this.provider.getSessionServer().publicKey);
    }

    public synchronized CompletableFuture<@Nullable JWebToken> getTokenAsync() {
        if (this.provider == null) {
            throw new IllegalStateException("A SessionProvider has not been set yet.");
        }
        if (this.isOffline()) {
            return CompletableFuture.completedFuture(null);
        }
        if (this.tokenFuture != null) {
            if (!this.tokenFuture.isDone() || this.isValid()) {
                return this.tokenFuture;
            }
            this.tokenFuture = null;
        }
        if (this.isValid()) {
            this.tokenFuture = CompletableFuture.completedFuture(this.getToken());
            return this.tokenFuture;
        }
        this.tokenFuture = CompletableFuture.supplyAsync(() -> {
            try {
                this.validate();
                if (!this.isValid()) {
                    this.provider.errorLog("Got invalid future after validate?", new Object[0]);
                    return null;
                }
                return this.getToken();
            }
            catch (Throwable ex) {
                this.provider.errorLog("Failed to validate session.", ex);
                return null;
            }
        }, this.SESSION_EXECUTOR);
        return this.tokenFuture;
    }

    public void validate() throws SessionValidationException {
        if (this.provider == null) {
            throw new SessionValidationException("A SessionProvider has not been set. Impossible to validate.");
        }
        if (this.isOffline()) {
            throw new SessionValidationException("Offline accounts can't obtain sessions.");
        }
        boolean didWork = false;
        Path sessionFile = this.getSessionFile();
        if (this.token == null && Files.exists(sessionFile, new LinkOption[0])) {
            try {
                this.token = JWebToken.parse(new String(Files.readAllBytes(sessionFile), StandardCharsets.UTF_8));
                didWork = true;
            }
            catch (IOException | InvalidJWebToken ex) {
                this.provider.warnLog("Failed to load session. New session will be obtained.", ex);
            }
        }
        if (!this.isValid() || this.token.isExpiredAt(TimeUnit.HOURS.toMillis(6L) + System.currentTimeMillis())) {
            String token;
            this.token = null;
            didWork = true;
            try {
                token = this.requestToken();
            }
            catch (IOException ex) {
                throw new SessionValidationException("Failed to acquire new session.", ex);
            }
            try {
                this.token = JWebToken.parse(token);
            }
            catch (InvalidJWebToken ex) {
                throw new SessionValidationException("New session token is corrupt?", ex);
            }
            try {
                if (Files.notExists(sessionFile.getParent(), new LinkOption[0])) {
                    Files.createDirectories(sessionFile.getParent(), new FileAttribute[0]);
                }
                Files.write(sessionFile, token.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            catch (IOException ex) {
                this.provider.warnLog("Failed to write session token to disk.", ex);
            }
        }
        if (!this.isValid()) {
            this.provider.errorLog("Don't have a valid token on normal exit of validate()", new Object[0]);
        }
        if (didWork) {
            this.onRefreshedCallbacks.forEach(e -> e.accept(this.getToken()));
        }
    }

    public JWebToken getToken() {
        if (this.token == null) {
            throw new IllegalStateException("Expected session to be validated.");
        }
        return this.token;
    }

    public void onTokenRefreshed(Consumer<JWebToken> cons) {
        this.onRefreshedCallbacks.add(cons);
    }

    public void forceResetToken() {
        this.token = null;
        if (this.tokenFuture != null) {
            if (!this.tokenFuture.isDone()) {
                this.tokenFuture.cancel(true);
            }
            this.tokenFuture = null;
        }
        try {
            Files.deleteIfExists(this.getSessionFile());
        }
        catch (IOException ex) {
            this.provider.warnLog("Failed to delete session file.", ex);
        }
    }

    private String requestToken() throws IOException {
        IOException error = null;
        for (int i = 0; i < 6; ++i) {
            try {
                String response = this.runTokenRequest(i < 3);
                AuthResponse resp = (AuthResponse)GSON.fromJson(response, AuthResponse.class);
                if ("success".equals(resp.status)) {
                    return resp.token;
                }
                throw new IOException("Failed session token request. Got API Error: " + resp.message);
            }
            catch (IOException ex) {
                if (error == null) {
                    error = ex;
                    continue;
                }
                error.addSuppressed(ex);
                continue;
            }
        }
        throw error;
    }

    private String runTokenRequest(boolean crypto) throws IOException {
        String endpoint;
        assert (this.provider != null);
        UUID uuid = this.provider.getUUID();
        if (uuid == null) {
            throw new IllegalStateException("UUID null? What?");
        }
        AuthRequest request = new AuthRequest();
        if (crypto) {
            endpoint = "/api/v1/auth/crypto";
            ProfileKeyPairResponse profileKeyPair = this.provider.getProfileKeyPair();
            if (profileKeyPair == null) {
                throw new IOException("Did not get valid profile keypair info.");
            }
            byte[] publicKey = PEMUtils.loadPem(profileKeyPair.keyPair.publicKey);
            PrivateKey privateKey = RSAUtils.loadRSAPrivateKeyPem(profileKeyPair.keyPair.privateKey);
            byte[] nonce = Crypto.generateNonce();
            byte[] signedNonce = RSAUtils.sign("SHA1withRSA", nonce, privateKey);
            request.minecraftCrypto = new AuthRequest.MinecraftCryptoAuth(this.provider.getUsername(), uuid, Crypto.base64Encode(publicKey), Instant.parse(profileKeyPair.expiresAt).toEpochMilli(), profileKeyPair.publicKeySignatureV2, Crypto.base64Encode(nonce), Crypto.base64Encode(signedNonce));
        } else {
            endpoint = "/api/v1/auth";
            String serverId = this.provider.beginAuth();
            if (serverId == null) {
                throw new IOException("Did not get a valid server id. ");
            }
            request.minecraft = new AuthRequest.MinecraftAuth(this.provider.getUsername(), uuid, serverId);
        }
        byte[] body = GSON.toJson((Object)request).getBytes(StandardCharsets.UTF_8);
        HttpUtils.Response response = HttpUtils.apiRequest(this.provider.getSessionServer().host + endpoint, "PUT", body, HttpUtils.mapOf("Content-Type", "application/json", "Content-Length", String.valueOf(body.length), "User-Agent", "Java/" + System.getProperty("java.version") + " MineTogetherSessions (" + this.provider.describe() + ")"));
        return response.body();
    }

    private Path getSessionFile() {
        assert (this.provider != null);
        UUID uuid = this.provider.getUUID();
        return this.sessionStorage.resolve(uuid + ".session");
    }

    public static final class SessionServer {
        public static final SessionServer DEFAULT = SessionServer.create("https://sessions.minetogether.io", SessionServer.class.getResourceAsStream("/default-public.pem"));
        public final String host;
        public final PublicKey publicKey;

        public SessionServer(String host, PublicKey publicKey) {
            this.host = host.endsWith("/") ? host.substring(0, host.length() - 1) : host;
            this.publicKey = publicKey;
        }

        public static SessionServer create(String host, Path pemFile) {
            try {
                return SessionServer.create(host, Files.newInputStream(pemFile, new OpenOption[0]));
            }
            catch (IOException ex) {
                throw new RuntimeException(ex);
            }
        }

        public static SessionServer create(String host, @WillClose @Nullable InputStream pemStream) {
            try {
                return new SessionServer(host, ECUtils.loadPublicKeyPem(Objects.requireNonNull(pemStream)));
            }
            catch (IOException ex) {
                throw new RuntimeException(ex);
            }
        }

        public static SessionServer create(String host, byte[] pemBytes) {
            try {
                return new SessionServer(host, ECUtils.loadPublicKeyPem(pemBytes));
            }
            catch (IOException ex) {
                throw new RuntimeException(ex);
            }
        }
    }
}

